// Copyright 2014 Google Inc. All Rights Reserved.

#include "AudioSource.h"

void AudioSource::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    MediaSourceService* ms = service->mutable_media_source_service();
    ms->set_available_type((MediaCodecType) mMediaCodecType);

    AudioConfiguration* ac = ms->mutable_audio_config();
    ac->set_sampling_rate(mSamplingRate);
    ac->set_number_of_bits(mNumberOfBits);
    ac->set_number_of_channels(mNumberOfChannels);
}

void AudioSource::setConfig(int samplingRate, int numberOfBits, int numberOfChannels) {
    mSamplingRate = samplingRate;
    mNumberOfBits = numberOfBits;
    mNumberOfChannels = numberOfChannels;
}

int AudioSource::routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case MEDIA_MESSAGE_MICROPHONE_REQUEST: {
            MicrophoneRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleMicrophoneRequest(req);
            }
            break;
        }
        case MEDIA_MESSAGE_ACK: {
            Ack ack;
            if (PARSE_PROTO(ack, ptr, len)) {
                handleAck(ack);
                ret = STATUS_SUCCESS;
            }
            break;
        }
    }
    return ret;
}

void AudioSource::send(uint64_t timestamp, const shared_ptr<IoBuffer>& data) {
    IoBuffer buf(data->size() + sizeof(uint16_t) + sizeof(uint64_t));
    uint8_t* ptr = (uint8_t*) buf.raw();
    WRITE_BE16_PTR_INCR(ptr, ((uint16_t) MEDIA_MESSAGE_DATA));
    WRITE_BE64_PTR_INCR(ptr, timestamp);
    memcpy(ptr, data->raw(), data->size());
    queueOutgoing(buf.raw(), buf.size());
}

void AudioSource::microphoneResponse(int32_t status, int32_t sessionId) {
    MicrophoneResponse resp;
    resp.set_status(status);
    if (status == STATUS_SUCCESS) {
        resp.set_session_id(sessionId);
    }
    IoBuffer buf;
    mRouter->marshallProto(MEDIA_MESSAGE_MICROPHONE_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

void AudioSource::handleAck(const Ack& ack) {
    mCallbacks->ackCallback(ack.session_id(), ack.ack());
}

int AudioSource::handleMicrophoneRequest(const MicrophoneRequest &req) {
    bool open = req.open();
    int32_t ret = 0;
    if (open) {
        ret = mCallbacks->microphoneRequestCallback(req.open(),
            req.anc_enabled(), req.ec_enabled(), req.max_unacked());
    } else {
        ret = mCallbacks->microphoneRequestCallback(open, false, false, 0);
    }

    if (ret < 0) {
        microphoneResponse(ret, 0);
        return STATUS_INTERNAL_ERROR;
    }
    microphoneResponse(STATUS_SUCCESS, ret);
    return STATUS_SUCCESS;
}
